/*
 * Copyright (c) Contributors, http://opensimulator.org/
 * See CONTRIBUTORS.TXT for a full list of copyright holders.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the OpenSimulator Project nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE DEVELOPERS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using System.IO;
using System.Net;
using OpenSim.Framework.ServiceAuth;

namespace OpenSim.Framework.Servers.HttpServer
{
    /// <summary>
    /// Base streamed request handler.
    /// </summary>
    /// <remarks>
    /// Inheriting classes should override ProcessRequest() rather than Handle()
    /// </remarks>
    public abstract class BaseStreamHandler : BaseRequestHandler, IStreamedRequestHandler
    {
        protected IServiceAuth m_Auth;

        protected BaseStreamHandler(string httpMethod, string path) : this(httpMethod, path, null, null) { }

        protected BaseStreamHandler(string httpMethod, string path, string name, string description)
            : base(httpMethod, path, name, description) {}

        protected BaseStreamHandler(string httpMethod, string path, IServiceAuth auth)
            : base(httpMethod, path, null, null) 
        {
            m_Auth = auth;
        }

        public virtual byte[] Handle(
            string path, Stream request, IOSHttpRequest httpRequest, IOSHttpResponse httpResponse)
        {
            RequestsReceived++;

            if (m_Auth != null)
            {
                HttpStatusCode statusCode;

                if (!m_Auth.Authenticate(httpRequest.Headers, httpResponse.AddHeader, out statusCode))
                {                
                    httpResponse.StatusCode = (int)statusCode;
                    httpResponse.ContentType = "text/plain";
                    return new byte[0];
                }
            }

            byte[] result = ProcessRequest(path, request, httpRequest, httpResponse);

            RequestsHandled++;

            return result;
        }

        protected virtual byte[] ProcessRequest(
            string path, Stream request, IOSHttpRequest httpRequest, IOSHttpResponse httpResponse)
        {
            return null;
        }
    }
}